#region References

using System;
using System.Data;
using System.Collections;
using System.Reflection;

using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;
using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.HL7InterfaceConfig;

#endregion

namespace gov.va.med.vbecs.BOL
{	
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// Aggregating wrapper for HL7Interface providing data validation and updating 
	/// interface control parameters config in database.
	/// </summary>

	#endregion

	public class HL7InterfaceConfig : BaseBusinessObject
	{
		private HL7Interface _interface;

		/// <summary>
		/// Builds an instance of the class around supplied instance of <see cref="HL7Interface"/>.
		/// </summary>
		/// <param name="hl7Inteface">Source interface to configure.</param>
		protected HL7InterfaceConfig( HL7Interface hl7Inteface ) : base()
		{ 
			BindToInterface( hl7Inteface );
		}

		/// <summary>
		/// Binds config wrapper to the HL7 interface.  
		/// </summary>
		/// <param name="hl7Inteface">HL7 interface to bind to.</param>
		protected void BindToInterface( HL7Interface hl7Inteface )
		{
			if( hl7Inteface == null )
				throw( new ArgumentNullException( "hl7Inteface" ) );

			this.SetInitialBrokenRules( BrRules.FormClassName, BrRules.RuleSets.BaseRuleSet );

			_interface = hl7Inteface;
			_interface.InterfaceParameterChanged += new EventHandler( AggregatedInterfaceParameterChanged );

			ValidateInterface();

			this.IsDirty = false;
			this.IsNew = false;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/1/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6777"> 
		///		<ExpectedInput>Updated interface config.</ExpectedInput>
		///		<ExpectedOutput>Interface config restored from DB.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6778"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Reloads bound interface config from DB. 
		/// </summary>
		public void ReloadInterfaceFromDb()
		{
			_interface.InterfaceParameterChanged -= new EventHandler( AggregatedInterfaceParameterChanged );
			// 
			// CR 2962
			BindToInterface( new HL7Interface(_interface.InterfaceName) );
		}

		/// <summary>
		/// The method is derived from base class but not supported because there is no need for it.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			throw( new NotSupportedException() );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4781"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4782"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>NotSupportedException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The method is derived from base class but not supported because there is no need for it.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			throw( new NotSupportedException() );
		}
		
		/// <summary>
		/// Event handled triggered by aggregated interface whenever any property is changed.
		/// </summary>
		/// <param name="sender">Standard sender object.</param>
		/// <param name="e">Standard event arguments object.</param>
		private void AggregatedInterfaceParameterChanged( object sender, EventArgs e )
		{
			IsDirty = true;

			ValidateInterface();			
		}

		/// <summary>
		/// Performs validation of aggregated interface setting broken rules.
		/// </summary>
		protected virtual void ValidateInterface()
		{
			// MBR_4.08
			RuleBroken( BrRules.InterfaceAdministratorEmailInvalid, 
				!RuleBroken( BrRules.InterfaceAdministratorEmailNotSet, StNullConvert.IsNull( _interface.InterfaceAdministratorEmail ) )
				&& !RegularExpressions.EmailAddress().IsMatch( _interface.InterfaceAdministratorEmail ) );
			//
			// MBR_4.10
			RuleBroken( BrRules.PurgeUnSuccessfulMessageOutOfRange, 
				!RuleBroken( BrRules.PurgeUnSuccessfulMessageNotSet, StNullConvert.IsNull( _interface.PurgeUnSuccessfulMessage ) )
				&& ( _interface.PurgeUnSuccessfulMessage < 1 || _interface.PurgeUnSuccessfulMessage > 30 ) );
			//
			// MBR_4.10
			RuleBroken( BrRules.PurgeSuccessfulMessageOutOfRange, 
				!RuleBroken( BrRules.PurgeSuccessfulMessagesNotSet, StNullConvert.IsNull( _interface.PurgeSuccessfulMessage ) )
				&& ( _interface.PurgeSuccessfulMessage < 1 || _interface.PurgeSuccessfulMessage > 30 ) );
			//
			RuleBroken( BrRules.ReTransmitAttemptsOutOfRange, 
				!RuleBroken( BrRules.ReTransmitAttemptsNotSet, StNullConvert.IsNull( _interface.ReTransmitAttempts ) )
				&& ( _interface.ReTransmitAttempts < 1 || _interface.ReTransmitAttempts > 99 ) );
			//
			RuleBroken( BrRules.AckTimeoutOutOfRange, 
				!RuleBroken( BrRules.AckTimeoutOutOfRange, StNullConvert.IsNull( _interface.AckTimeout ) )
				&& ( _interface.AckTimeout < 1 || _interface.AckTimeout > 999 ) );
			//
			// MBR_1.02
			RuleBroken( BrRules.VbecsPortNumberOutOfRange,
				!RuleBroken( BrRules.VbecsPortNumberNotSet, StNullConvert.IsNull( _interface.VbecsPortNumber ) )
				&& ( _interface.VbecsPortNumber < 1024 || _interface.VbecsPortNumber > 65535 ) );
			//
			// MBR_4.03
			RuleBroken( BrRules.InterfaceFacilityIdsMatch,
				!StNullConvert.IsNull( _interface.InterfaceFacilityId ) && 
				!StNullConvert.IsNull( _interface.VbecsFacilityId ) && 
				( _interface.InterfaceFacilityId.ToLower() == _interface.VbecsFacilityId.ToLower() ) );
			//
			RuleBroken( BrRules.VbecsIPAddressNotSet, StNullConvert.IsNull( _interface.VbecsIPAddress ) );		
			//
			// CPRS and BCE COTS interfaces requires interface IP address & port number (CR 2962)
			bool ipAndPortNumberRequired = ( _interface.InterfaceName == InterfaceName.CPRS.ToString() ||
				_interface.InterfaceName == InterfaceName.BCE_COTS.ToString() );
			//
			// MBR_1.02
			// CR 2962 
			RuleBroken( BrRules.InterfacePortNumberOutOfRange,
				ipAndPortNumberRequired
				&& !RuleBroken( BrRules.InterfacePortNumberNotSet, ipAndPortNumberRequired 
				&& StNullConvert.IsNull( _interface.InterfacePortNumber ) )
				&& ( _interface.InterfacePortNumber < 1024 || _interface.InterfacePortNumber > 65535 ) );
			//
			// MBR 1.12
			RuleBroken( BrRules.InterfaceIPAddressAndDomainNameNotSpecified, 
				ipAndPortNumberRequired 
				&& StNullConvert.IsNull( _interface.InterfaceIPAddress ) 
				&&  (_interface.InterfaceDomainName == null 
				|| _interface.InterfaceDomainName.Trim() == String.Empty) );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4787"> 
		///		<ExpectedInput>ACK timeout is increased by one in aggregated interface.</ExpectedInput>
		///		<ExpectedOutput>ACK timeout is saved.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4788"> 
		///		<ExpectedInput>ACK timeout is set to -1 in aggregated interface.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves interface control parameters. 
		/// </summary>
		public void Save()
		{
			if( !IsDirty )
				return;

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToSaveInvalidBusinessObject( typeof( HL7InterfaceConfig ).Name ).ResString ) );

			InterfaceControl.UpdateInterface( _interface.ExportAsDataRow() );

			IsDirty = false;
			IsNew = false;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4785"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-empty list of HL7 inteface configs.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4786"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves configuration objects for all HL7 interfaces registered in the VBECS database.
		/// </summary>
		/// <returns>List of all HL7 interfaces in the VBECS DB.</returns>
		public static IList GetHL7InterfaceConfigs()
		{
			ArrayList _list = new ArrayList();

			foreach( HL7Interface _interface in HL7Interface.GetHL7Interfaces() )
				_list.Add( new HL7InterfaceConfig( _interface ) );

			return _list;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4783"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null reference to aggregated HL7 interface.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4784"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Aggregated HL7 interface.
		/// </summary>
		public HL7Interface Interface
		{
			get
			{
				return _interface;
			}
		}
	}
}
